import React, { useState } from "react";
import { Upload, File, Check, Copy, X, Cloud, ArrowUp } from "lucide-react";
import API from "../api";

export default function FileUpload() {
  const [file, setFile] = useState(null);
  const [downloadLink, setDownloadLink] = useState("");
  const [isUploading, setIsUploading] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [isDragging, setIsDragging] = useState(false);
  const [copied, setCopied] = useState(false);

  function handleFileSelect(selectedFile) {
    if (selectedFile) {
      setFile(selectedFile);
      setDownloadLink("");
      setUploadProgress(0);
    }
  }

  function handleDragOver(e) {
    e.preventDefault();
    setIsDragging(true);
  }

  function handleDragLeave(e) {
    e.preventDefault();
    setIsDragging(false);
  }

  function handleDrop(e) {
    e.preventDefault();
    setIsDragging(false);
    const droppedFile = e.dataTransfer.files[0];
    handleFileSelect(droppedFile);
  }

  function removeFile() {
    setFile(null);
    setDownloadLink("");
    setUploadProgress(0);
  }

  async function handleUpload(e) {
    e.preventDefault();
    if (!file) return alert("Please select a file");

    setIsUploading(true);
    setUploadProgress(0);

    // Simulate upload progress for demo
    const interval = setInterval(() => {
      setUploadProgress((prev) => {
        if (prev >= 90) {
          clearInterval(interval);
          return 90;
        }
        return prev + 10;
      });
    }, 200);

    setTimeout(() => {
      clearInterval(interval);
      setUploadProgress(100);
      setDownloadLink("https://example.com/files/abc123xyz");
      setIsUploading(false);
      alert("File uploaded successfully!");
    }, 2500);

    const formData = new FormData();
    formData.append("file", file);

    try {
      const res = await API.post("/file/upload", formData, {
        headers: { "Content-Type": "multipart/form-data" },
      });
      setDownloadLink(res.data.downloadLink);
      setUploadProgress(100);
      alert("File uploaded successfully!");
    } catch (err) {
      alert(err.response?.data?.message || "Upload failed");
    } finally {
      setIsUploading(false);
    }
  }

  function copyToClipboard() {
    navigator.clipboard.writeText(downloadLink);
    setCopied(true);
    setTimeout(() => setCopied(false), 2000);
  }

  function formatFileSize(bytes) {
    if (!bytes) return "0 Bytes";
    const k = 1024;
    const sizes = ["Bytes", "KB", "MB", "GB"];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return Math.round((bytes / Math.pow(k, i)) * 100) / 100 + " " + sizes[i];
  }

  return (
    <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-indigo-100 via-purple-50 to-pink-100 p-4">
      <div className="w-full max-w-2xl">
        {/* Decorative elements */}
        <div className="absolute top-0 left-0 w-72 h-72 bg-purple-300 rounded-full mix-blend-multiply filter blur-xl opacity-70 animate-blob"></div>
        <div className="absolute top-0 right-0 w-72 h-72 bg-indigo-300 rounded-full mix-blend-multiply filter blur-xl opacity-70 animate-blob animation-delay-2000"></div>
        <div className="absolute bottom-0 left-20 w-72 h-72 bg-pink-300 rounded-full mix-blend-multiply filter blur-xl opacity-70 animate-blob animation-delay-4000"></div>

        {/* Upload card */}
        <div className="relative bg-white/80 backdrop-blur-lg rounded-3xl shadow-2xl p-8 border border-white/20">
          {/* Logo/Icon */}
          <div className="flex justify-center mb-6">
            <div className="w-16 h-16 bg-gradient-to-br from-indigo-600 to-purple-600 rounded-2xl flex items-center justify-center shadow-lg">
              <Cloud className="w-8 h-8 text-white" />
            </div>
          </div>

          {/* Header */}
          <div className="text-center mb-8">
            <h2 className="text-3xl font-bold bg-gradient-to-r from-indigo-600 to-purple-600 bg-clip-text text-transparent mb-2">
              Upload Your File
            </h2>
            <p className="text-gray-600 text-sm">
              Drag and drop or click to browse
            </p>
          </div>

          {/* Upload area */}
          <div
            className={`relative border-2 border-dashed rounded-2xl p-8 transition-all ${
              isDragging
                ? "border-indigo-500 bg-indigo-50"
                : "border-gray-300 hover:border-indigo-400 hover:bg-gray-50"
            }`}
            onDragOver={handleDragOver}
            onDragLeave={handleDragLeave}
            onDrop={handleDrop}
          >
            {!file ? (
              <div className="text-center">
                <div className="flex justify-center mb-4">
                  <div className="w-20 h-20 bg-gradient-to-br from-indigo-100 to-purple-100 rounded-2xl flex items-center justify-center">
                    <Upload className="w-10 h-10 text-indigo-600" />
                  </div>
                </div>
                <p className="text-lg font-semibold text-gray-700 mb-2">
                  Drop your file here
                </p>
                <p className="text-sm text-gray-500 mb-4">
                  or click to browse from your device
                </p>
                <label className="inline-block">
                  <input
                    type="file"
                    className="hidden"
                    onChange={(e) => handleFileSelect(e.target.files[0])}
                  />
                  <span className="inline-flex items-center gap-2 px-6 py-3 bg-gradient-to-r from-indigo-600 to-purple-600 text-white rounded-xl font-semibold hover:from-indigo-700 hover:to-purple-700 cursor-pointer transition-all shadow-lg hover:shadow-xl">
                    <File className="w-5 h-5" />
                    Choose File
                  </span>
                </label>
                <p className="text-xs text-gray-400 mt-4">
                  Max file size: 100MB
                </p>
              </div>
            ) : (
              <div className="space-y-4">
                {/* Selected file info */}
                <div className="flex items-center justify-between p-4 bg-gradient-to-r from-indigo-50 to-purple-50 rounded-xl border border-indigo-100">
                  <div className="flex items-center gap-3 flex-1 min-w-0">
                    <div className="w-12 h-12 bg-gradient-to-br from-indigo-600 to-purple-600 rounded-lg flex items-center justify-center flex-shrink-0">
                      <File className="w-6 h-6 text-white" />
                    </div>
                    <div className="flex-1 min-w-0">
                      <p className="font-semibold text-gray-700 truncate">
                        {file.name}
                      </p>
                      <p className="text-sm text-gray-500">
                        {formatFileSize(file.size)}
                      </p>
                    </div>
                  </div>
                  {!isUploading && !downloadLink && (
                    <button
                      onClick={removeFile}
                      className="ml-3 p-2 hover:bg-red-100 rounded-lg transition-colors"
                    >
                      <X className="w-5 h-5 text-red-500" />
                    </button>
                  )}
                </div>

                {/* Upload progress */}
                {isUploading && (
                  <div className="space-y-2">
                    <div className="flex justify-between text-sm">
                      <span className="text-gray-600">Uploading...</span>
                      <span className="text-indigo-600 font-semibold">
                        {uploadProgress}%
                      </span>
                    </div>
                    <div className="h-2 bg-gray-200 rounded-full overflow-hidden">
                      <div
                        className="h-full bg-gradient-to-r from-indigo-600 to-purple-600 transition-all duration-300"
                        style={{ width: `${uploadProgress}%` }}
                      ></div>
                    </div>
                  </div>
                )}

                {/* Upload button */}
                {!downloadLink && (
                  <button
                    onClick={handleUpload}
                    disabled={isUploading}
                    className="w-full bg-gradient-to-r from-indigo-600 to-purple-600 text-white py-3.5 rounded-xl font-semibold hover:from-indigo-700 hover:to-purple-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 transition-all shadow-lg hover:shadow-xl flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed"
                  >
                    {isUploading ? (
                      <>
                        <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                        Uploading...
                      </>
                    ) : (
                      <>
                        <ArrowUp className="w-5 h-5" />
                        Upload File
                      </>
                    )}
                  </button>
                )}
              </div>
            )}
          </div>

          {/* Download link section */}
          {downloadLink && (
            <div className="mt-6 p-6 bg-gradient-to-r from-green-50 to-emerald-50 rounded-2xl border border-green-200">
              <div className="flex items-start gap-3 mb-4">
                <div className="w-10 h-10 bg-green-500 rounded-lg flex items-center justify-center flex-shrink-0">
                  <Check className="w-6 h-6 text-white" />
                </div>
                <div className="flex-1">
                  <h3 className="font-semibold text-green-900 mb-1">
                    Upload Successful!
                  </h3>
                  <p className="text-sm text-green-700">
                    Your file is ready to share
                  </p>
                </div>
              </div>

              <div className="space-y-3">
                <div className="flex items-center gap-2 p-3 bg-white rounded-lg border border-green-200">
                  <input
                    type="text"
                    value={downloadLink}
                    readOnly
                    className="flex-1 text-sm text-gray-700 bg-transparent outline-none"
                  />
                  <button
                    onClick={copyToClipboard}
                    className="p-2 hover:bg-green-100 rounded-lg transition-colors"
                  >
                    {copied ? (
                      <Check className="w-5 h-5 text-green-600" />
                    ) : (
                      <Copy className="w-5 h-5 text-gray-600" />
                    )}
                  </button>
                </div>

                <div className="flex gap-3">
                  <a
                    href={downloadLink}
                    target="_blank"
                    rel="noreferrer"
                    className="flex-1 text-center py-2.5 bg-green-600 text-white rounded-lg font-medium hover:bg-green-700 transition-colors"
                  >
                    Open Link
                  </a>
                  <button
                    onClick={() => {
                      setFile(null);
                      setDownloadLink("");
                    }}
                    className="flex-1 text-center py-2.5 bg-white border border-green-600 text-green-600 rounded-lg font-medium hover:bg-green-50 transition-colors"
                  >
                    Upload Another
                  </button>
                </div>
              </div>
            </div>
          )}

          {/* Info section */}
          <div className="mt-6 grid grid-cols-3 gap-4 text-center">
            <div className="p-4 bg-gradient-to-br from-indigo-50 to-purple-50 rounded-xl">
              <p className="text-2xl font-bold text-indigo-600 mb-1">100MB</p>
              <p className="text-xs text-gray-600">Max Size</p>
            </div>
            <div className="p-4 bg-gradient-to-br from-indigo-50 to-purple-50 rounded-xl">
              <p className="text-2xl font-bold text-indigo-600 mb-1">Fast</p>
              <p className="text-xs text-gray-600">Upload Speed</p>
            </div>
            <div className="p-4 bg-gradient-to-br from-indigo-50 to-purple-50 rounded-xl">
              <p className="text-2xl font-bold text-indigo-600 mb-1">Secure</p>
              <p className="text-xs text-gray-600">Encrypted</p>
            </div>
          </div>
        </div>

        {/* Footer text */}
        <p className="text-center text-xs text-gray-500 mt-6">
          Files are encrypted and stored securely. Links expire after 7 days.
        </p>
      </div>
    </div>
  );
}
